document.addEventListener('DOMContentLoaded', () => {
  const form = document.getElementById('settings-form');
  const captchaBotEnabled = document.getElementById('captchaBotEnabled');
  const apiToken = document.getElementById('apiToken');
  const autoSubmitEnabled = document.getElementById('autoSubmitEnabled');
  const captchaDelay = document.getElementById('captchaDelay');

  // Default settings shown to the user
  const defaults = {
    captchaBotEnabled: true,
    apiToken: '',
    autoSubmitEnabled: true,
    captchaDelay: 500
  };

  // Load saved settings from chrome.storage
  chrome.storage.local.get(defaults, (items) => {
    captchaBotEnabled.checked = items.captchaBotEnabled;
    apiToken.value = items.apiToken || '';
    autoSubmitEnabled.checked = items.autoSubmitEnabled;
    captchaDelay.value = items.captchaDelay;
  });

  // Save settings on form submit
  form.addEventListener('submit', (e) => {
    e.preventDefault();
    // We get the existing settings first to avoid overwriting the hardcoded server URL
    chrome.storage.local.get(null, (existingSettings) => {
      const newSettings = {
        ...existingSettings, // Keep existing settings like localServerUrl
        captchaBotEnabled: captchaBotEnabled.checked,
        apiToken: apiToken.value.trim(),
        autoSubmitEnabled: autoSubmitEnabled.checked,
        captchaDelay: parseInt(captchaDelay.value, 10)
      };
      chrome.storage.local.set(newSettings, () => {
        const button = form.querySelector('button');
        const originalText = button.textContent;
        button.textContent = 'Saved!';
        setTimeout(() => {
          button.textContent = originalText;
          // window.close(); // Don't close immediately so they can see stats update
          fetchStats(newSettings.apiToken);
        }, 750);
      });
    });
  });

  // Initial stats fetch
  chrome.storage.local.get(['apiToken'], (items) => {
    if (items.apiToken) {
      fetchStats(items.apiToken);
    }
  });

  function fetchStats(token) {
    if (!token) return;

    const container = document.getElementById('stats-container');
    const usageStat = document.getElementById('usageStat');
    const limitStat = document.getElementById('limitStat');
    const expiryStat = document.getElementById('expiryStat');
    const statusMsg = document.getElementById('statusMessage');

    container.style.display = 'flex';
    usageStat.textContent = '...';

    // Default to localhost, or fetch from storage if you have a setting for it
    // For now we assume the remote server or localhost
    const SERVER_URL = "https://www.appservices.dev"; // Or localhost for testing

    fetch(`${SERVER_URL}/status`, {
      method: 'GET',
      headers: {
        'Authorization': `Bearer ${token}`
      }
    })
      .then(response => {
        if (!response.ok) {
          return response.json().then(err => { throw new Error(err.message || 'Error'); });
        }
        return response.json();
      })
      .then(data => {
        usageStat.textContent = data.daily_usage;
        limitStat.textContent = data.daily_limit;
        expiryStat.textContent = data.expiration_date;
        statusMsg.style.display = 'none';

        if (data.daily_usage >= data.daily_limit) {
          usageStat.style.color = '#e06c75'; // Red if limit reached
        } else {
          usageStat.style.color = '#98c379'; // Green otherwise
        }
      })
      .catch(error => {
        console.error("Stats error:", error);
        statusMsg.textContent = "Error: " + error.message;
        statusMsg.style.display = 'block';
        usageStat.textContent = '-';
        limitStat.textContent = '-';
        expiryStat.textContent = '-';
      });
  }
});