export async function injectCaptchaBot(tabId, settings) {
  try {
    await chrome.scripting.executeScript({
      target: { tabId },
      world: 'MAIN',
      args: [settings],
      func: (config) => {
        // Prevent multiple injections on the same page
        if (window.captchaBotRunning) return;
        window.captchaBotRunning = true;

        (async () => {

          // Add custom styles to override the website's default selection styles.
          const blockingStyle = document.createElement('style');
          blockingStyle.textContent = `
              /* Block any existing img-selected styles with maximum specificity */
              .img-selected,
              body .img-selected,
              html body .img-selected,
              * .img-selected {
                  border: 5px solid #1d99ff !important;
              }
          `;
          document.head.appendChild(blockingStyle);

          // Flag to prevent the solver from running more than once
          let isSolving = false;

          // --- Notification Manager for user feedback ---
          class NotificationManager {
            createStyles() {
              if (document.getElementById('bls-notification-styles')) return;
              const style = document.createElement('style');
              style.id = 'bls-notification-styles';
              style.textContent = `
                .bls-notification {
                    position: fixed; top: 20px; right: 20px; padding: 12px 20px;
                    border-radius: 6px; color: white; font-family: Arial, sans-serif;
                    font-size: 14px; z-index: 10001; box-shadow: 0 4px 12px rgba(0,0,0,0.3);
                    transition: opacity 0.3s ease, transform 0.3s ease; transform: translateX(100%); opacity: 0;
                }
                .bls-notification.show { transform: translateX(0); opacity: 1; }
                .bls-notification.success { background-color: #28a745; }
                .bls-notification.error { background-color: #dc3545; }
                .bls-notification.processing { background-color: #007bff; }
              `;
              document.head.appendChild(style);
            }

            showNotification(message, type = 'info', duration = 4000) {
              this.createStyles();
              document.querySelectorAll('.bls-notification').forEach(el => el.remove());
              const notification = document.createElement('div');
              notification.className = `bls-notification ${type}`;
              notification.textContent = message;
              document.body.appendChild(notification);
              setTimeout(() => notification.classList.add('show'), 50);
              if (duration > 0) {
                setTimeout(() => {
                  notification.classList.remove('show');
                  setTimeout(() => notification.remove(), 300);
                }, duration);
              }
              return notification;
            }
            showSuccess(message) { return this.showNotification(message, 'success'); }
            showError(message) { return this.showNotification(message, 'error'); }
            showProcessing(message) { return this.showNotification(message, 'processing', 0); }
          }

          const notificationManager = new NotificationManager();

          const clearPreviousSelections = () => {
            document.querySelectorAll('.captcha-img').forEach(img => {
              img.style.filter = '';
              img.style.transition = '';
              img.style.border = '';
              img.style.boxShadow = '';
            });
          };

          const applyFogEffect = (imageElement) => {
            if (imageElement) {
              imageElement.style.transition = 'filter 0.3s ease, border 0.3s ease, box-shadow 0.3s ease';
              imageElement.style.filter = 'blur(2px) opacity(1.9)';
            }
          };

          const getCaptchaTarget = () => {
            const labels = Array.from(document.querySelectorAll(".box-label"));
            if (labels.length === 0) return "";
            const sortedLabels = labels.sort((a, b) => (parseInt(getComputedStyle(b).zIndex) || 0) - (parseInt(getComputedStyle(a).zIndex) || 0));
            return sortedLabels[0].textContent.replace(/\D+/, '');
          };

          const getCaptchaGrid = () => {
            const captchaContainers = Array.from(document.querySelectorAll(".captcha-img")).map(img => img.parentElement).filter(el => el && getComputedStyle(el).display !== "none");
            if (captchaContainers.length === 0) return [];
            const groupedByRow = captchaContainers.reduce((acc, element) => {
              const row = Math.floor(element.offsetTop);
              if (!acc[row]) acc[row] = [];
              acc[row].push(element);
              return acc;
            }, {});
            return Object.values(groupedByRow).flatMap(row => {
              const sortedRow = row.sort((a, b) => (parseInt(getComputedStyle(b).zIndex) || 0) - (parseInt(getComputedStyle(a).zIndex) || 0));
              return sortedRow.slice(0, 3).sort((a, b) => a.offsetLeft - b.offsetLeft).map(el => el.querySelector(".captcha-img"));
            }).filter(Boolean);
          };

          const applySolution = (solution, captchaTarget, captchaGrid) => {
            Object.entries(solution).forEach(([index, extractedNumber]) => {
              if (extractedNumber === captchaTarget) {
                const imageElement = captchaGrid[index];
                if (imageElement) {
                  imageElement.click();
                  // Apply both the border and the new fog effect for clear feedback
                  imageElement.style.border = '';
                  imageElement.style.boxShadow = '';
                  applyFogEffect(imageElement);
                }
              }
            });

            if (config.autoSubmitEnabled) {
              const btnVerify = document.getElementById("btnVerify");
              const captchaDelay = config.captchaDelay || 0;
              setTimeout(() => btnVerify?.click(), captchaDelay);
            }
          };

          const solveWithLocalServer = async (captchaImages, captchaTarget, captchaGrid) => {
            if (!config.localEnabled) {
              notificationManager.showError('Local solver is disabled in settings.');
              return;
            }

            if (!config.apiToken || config.apiToken.trim() === '') {
              notificationManager.showError('API Token is required. Please configure it in the extension settings.');
              return;
            }

            const processingNotice = notificationManager.showProcessing('Solving captcha...');
            try {
              const response = await fetch(config.localServerUrl, {
                method: "POST",
                headers: {
                  "Content-Type": "application/json",
                  "Authorization": `Bearer ${config.apiToken}`
                },
                body: JSON.stringify({ target: captchaTarget, images: captchaImages }),
                signal: AbortSignal.timeout(10000)
              });
              processingNotice.remove();
              if (!response.ok) {
                if (response.status === 401) {
                  throw new Error('Invalid or expired API token');
                }
                throw new Error(`Server error: ${response.status}`);
              }
              const data = await response.json();
              if (data.solution) {
                notificationManager.showSuccess('Captcha solved!');
                applySolution(data.solution, captchaTarget, captchaGrid);
              } else {
                throw new Error('Solver failed to find a solution.');
              }
            } catch (error) {
              processingNotice.remove();
              notificationManager.showError(`Error: ${error.message}`);
            }
          };

          const runSolver = async () => {
            // If a solve is already in progress, exit immediately.
            if (isSolving) {
              return;
            }

            const captchaTarget = getCaptchaTarget();
            const captchaGrid = getCaptchaGrid();

            // Only proceed if the captcha elements are actually visible.
            if (captchaGrid.length === 0 || !captchaTarget) {
              return; // Captcha not ready yet.
            }

            // Set the flag to true to block any subsequent calls.
            isSolving = true;

            // Clear visual styles from any previous attempts
            clearPreviousSelections();

            const captchaImages = Object.fromEntries(captchaGrid.map((img, index) => [index, img.src]));
            await solveWithLocalServer(captchaImages, captchaTarget, captchaGrid);
          };

          // Use a MutationObserver to detect when the captcha appears on the page.
          const observer = new MutationObserver((mutations, obs) => {
            if (document.querySelector('.captcha-img')) {
              runSolver();
              // We can disconnect here because we only need to solve it once.
              obs.disconnect();
            }
          });

          observer.observe(document.body, { childList: true, subtree: true });

          // Also run the solver once at the start, in case the captcha is already present.
          runSolver();
        })();
      },
    });
  } catch (error) {
    console.error('Failed to inject captcha script:', error);
  }
}