/**
 * Captcha Solver Extension - Background Service Worker
 *
 * Responsibilities:
 * 1. Initialize default settings on first installation.
 * 2. Set up a periodic alarm to check the local server's health.
 * 3. Listen for tab updates to detect when a target page is loaded.
 * 4. Inject the content script (`Captcha.js`) onto target pages if the bot is enabled.
 * 5. Perform the server health check when the alarm fires or the browser starts.
 */

import { injectCaptchaBot } from './content_scripts/Captcha.js';

// --- 1. Installation & Initialization ---

// This listener runs only when the extension is first installed or updated.
// This listener runs only when the extension is first installed or updated.
chrome.runtime.onInstalled.addListener(() => {
  // Define the default settings for the extension.
  const defaultSettings = {
    captchaBotEnabled: true,
    localEnabled: true, // This is now hardcoded
    localServerUrl: 'https://www.appservices.dev/solve',
    apiToken: '', // API token for authentication
    autoSubmitEnabled: true,
    captchaDelay: 500
  };

  // Store the default settings. This prevents errors on the first run.
  chrome.storage.local.set(defaultSettings);
  console.log('Captcha Solver: Default settings have been initialized.');

  // Create a periodic alarm to check the health of the local server.
  // This is the correct place to create alarms, ensuring it's done only once.
  chrome.alarms.create('keepAlive', {
    delayInMinutes: 1,  // Wait 1 minute before the first check.
    periodInMinutes: 1  // Repeat the check every 1 minute.
  });
  console.log('Captcha Solver: Keep-alive alarm for server health check has been created.');
});


// --- 2. Content Script Injection Logic ---

// An array of URL paths where the captcha solver should be active.
// Using .includes() for matching makes it flexible.
const targetPaths = [
  '/appointment/appointmentcaptcha',
  '/newcaptcha/logincaptcha',
  '/appointment/newappointment',
];

// This listener fires whenever a tab is updated (e.g., new page, reload).
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  // We only want to act when the page has finished loading.
  if (changeInfo.status !== 'complete' || !tab.url || !tab.url.startsWith('http')) {
    return;
  }

  try {
    const url = new URL(tab.url);
    const currentPath = url.pathname.toLowerCase();

    // Check if the current page's path matches any of our target paths.
    const isTargetPath = targetPaths.some(targetPath => currentPath.includes(targetPath));

    if (isTargetPath) {
      console.log(`Captcha Solver: Target path matched on ${tab.url}. Checking settings...`);

      // Retrieve the user's settings from storage.
      chrome.storage.local.get(null, (settings) => {
        // Inject the content script only if the bot is enabled in the settings.
        if (settings && settings.captchaBotEnabled) {
          console.log('Captcha Solver: Bot is enabled. Injecting content script.');
          injectCaptchaBot(tabId, settings);
        } else {
          console.log('Captcha Solver: Bot is disabled in settings. No script injected.');
        }
      });
    }
  } catch (error) {
    console.error("Captcha Solver: An error occurred in the tabs.onUpdated listener:", error);
  }
});


// --- 3. Server Health Check (Keep-Alive) ---

/**
 * Asynchronously checks the /health endpoint of the local server.
 */
async function performHealthCheck() {
  try {
    const { localServerUrl, localEnabled } = await chrome.storage.local.get(['localServerUrl', 'localEnabled']);

    // Don't perform the check if the local server feature is disabled.
    if (!localEnabled || !localServerUrl) {
      return;
    }

    const healthUrl = new URL(localServerUrl);
    healthUrl.pathname = '/health'; // The endpoint for health checks.

    const response = await fetch(healthUrl.toString(), { signal: AbortSignal.timeout(5000) });

    if (!response.ok) {
      throw new Error(`Server responded with status: ${response.status}`);
    }

    const data = await response.json();
    if (data.status === 'healthy') {
      console.log('Keep-alive: Local captcha server is healthy.');
    } else {
      console.warn('Keep-alive: Local captcha server reported an unhealthy status.');
    }
  } catch (error) {
    console.error('Keep-alive: Local captcha server is unreachable.', error.message);
  }
}

// This listener fires when the 'keepAlive' alarm we created goes off.
chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === 'keepAlive') {
    performHealthCheck();
  }
});

// This listener runs when the browser first starts up.
chrome.runtime.onStartup.addListener(() => {
  console.log('Captcha Solver: Browser startup detected. Performing initial health check.');
  performHealthCheck();
});